%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Codes for Example 2.7
% Basic MCMC to estimate diffusion coefficient of an 
% Ornstein-Uhlenbeck process
% Created by Shixiao Jiang, edited by John Harlim 
% Last edited: March 16, 2018  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all, clc, close all
% length of observation data
ObsTest_Length = 11; % % length of obs + 1

% MCMC Parameters
C = 0.05; % tunable parameter in the Metropolis scheme
Rho0 = 7; % initial value of MCMC
Rho_Length = 300000; % # of points in MCMC

% prior density parameters - inverse gamma distribution
kalpha = 3; % parameter of prior density 
kbeta = 10.0; % parameter of prior density


%% create obseration data - well-sampled Gaussian
bsq_true = 5; % b^2 true value
x_true_uniform=1/(ObsTest_Length):1/(ObsTest_Length):1-1/(ObsTest_Length); % uniform
x_true_gaussian = norminv(x_true_uniform,0,1); % well-sampled standard gaussian
zvari_T = (x_true_gaussian'-0)*bsq_true.^0.5; % mean zero, variance bsq_true, gaussian

%% intialize MCMC points
Rho_MCMC = zeros(Rho_Length,1);
Rho_MCMC(1,1) = Rho0;

%% compute likelihood function pz_newz_newrho_ONE
pz_newz_newrho_ONE=1/sqrt(Rho_MCMC(1))*exp(-zvari_T.^2/2/Rho_MCMC(1));

% Bayesian - posterior
p_allz_rho = zeros(Rho_Length,1);

%%% prior density
prior0_rho = Rho_MCMC(1).^(-kalpha-1).*exp(-kbeta./Rho_MCMC(1));
%%% log of posterior
p_allz_rho(1) = sum(log(abs(pz_newz_newrho_ONE))) + log(prior0_rho);


%% MCMC simulation
for ii=1:Rho_Length-1 

    %%% give a proposal of parameter
    rho_test = Rho_MCMC(ii)+C^0.5*randn;
       
        
    %%% compute likelihood function, pz_newz_newrho_ONE
    pz_newz_newrho_ONE=1/sqrt(rho_test).*exp(-zvari_T.^2/2/rho_test);
       

    %%% prior density
    prior0_rho = rho_test.^(-kalpha-1).*exp(-kbeta./rho_test);
    %%% log of posterior
    p_allz_rho(ii+1) = sum(log(abs(pz_newz_newrho_ONE))) + log(prior0_rho); % prod(abs(pz_newz_newrho))*prior0_rho;
        

    if log(rand(1,1))<(p_allz_rho(ii+1)-p_allz_rho(ii))    % accept proposal or not
        Rho_MCMC(ii+1) = rho_test; 
    else
        Rho_MCMC(ii+1) = Rho_MCMC(ii);
    end

end


%% plot figure 2.1

figure(1)
hold on

new_alpha = kalpha+0.5*length(zvari_T); % parameter of analytic posterior
new_beta = kbeta+0.5*sum(zvari_T.^2); % parameter of analytic posterior
xx=0.2:0.05:10; % x axis - b^2 value

% analytic posterior
if new_alpha>1
    series = new_alpha-1:-1:1e-10;
    log_gamma_new_alpha = sum(log(series)) + log(gamma(series(end)));
end
zz_2 = new_alpha*log(new_beta)-log_gamma_new_alpha+(-new_alpha-1)*log(xx)+(-1./xx.* new_beta);
zz_2_exp = exp(zz_2);
h_analytic2 = plot(xx,zz_2_exp,'k--','linewidth',2);

grey = [.6,.6,.6];
range_b2 = max(Rho_MCMC) - min(Rho_MCMC);
Rho_min = min(Rho_MCMC)-range_b2/10;
Rho_max = max(Rho_MCMC)+range_b2/10;
[bandwidth_Rho,density_Rho,Rhomesh,cdf_Rho]=kde(Rho_MCMC(51:end));
plot(Rhomesh,density_Rho,'color',grey,'linewidth',2)


% the mean of analytic posterior
if new_alpha>1
    plot(new_beta/(new_alpha-1),0,'ko','markersize',8,'linewidth',2,'MarkerFaceColor','k');
end

% Mean of MCMC posterior
plot(mean(Rho_MCMC(51:end)),0,'s','color',grey,'LineWidth',2,'markersize',8); 

% true value of parameter
plot(bsq_true,0,'k*','markersize',8,'LineWidth',2);

% set figure property
set(gca,'position',[0.12,0.15,0.7,0.8]);
set(gca,'FontSize',16, 'FontName','Arial')
xlabel('parameter $b^{2}$','fontsize',16,'FontName','Times New Roman','Interpreter','latex');
ylabel('posterior $p(b^{2}|x)$','fontsize',16,'FontName','Times New Roman','Rotation',90,'Interpreter','latex');
set(gca,'xtick',0:2:14)
axis([0 14 0 0.5])
legend('Analytic posterior','MCMC posterior','analytic post mean','MCMC post mean','truth')









